<?php

namespace App\Traits;

use Carbon\Carbon;
use App\Models\HeadGroup;
use App\Models\IncomeHead;
use App\Models\AccountType;
use App\Models\AccountingYear;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Auth;
use App\Models\AccountBroughtForward;
use App\Models\PaymentType;
use App\Models\Bank;

trait CommonTrait
{
    public function getPreviousFiscalYearShort()
    {
        $today = Carbon::today();
        $fiscalStartMonth = 4; // April

        if ($today->month >= $fiscalStartMonth) {
            $prevYearStart = $today->year - 1;
        } else {
            $prevYearStart = $today->year - 2;
        }

        $prevYearEnd = $prevYearStart + 1;
        // Return short format: last two digits
        return substr($prevYearStart, -2) . '-' . substr($prevYearEnd, -2);
    }

    public function getAccountType($divition_flag = null)
    {
        $query = AccountType::select('account_type_id', 'organisation_id', 'account_type_Name', 'divition_flag')
            ->where('status', 'active');

        // Apply condition only if $divition_flag is present
        if (!empty($divition_flag)) {
            $query->where('divition_flag', $divition_flag);
        }

        $accountTypes = $query->orderBy('account_type_id', 'DESC')->get();

        return $accountTypes;
    }


    public function getIncomeHead($flag = null)
    {
        $query = IncomeHead::select('*')
            ->where(['status' => 'active']);
        // Apply condition only if $flag is present
        if (!empty($flag)) {
            $query->where('income_ac_type_flg', $flag);
        }
        $income_heads = $query->orderBy('sort_order', 'ASC')->get();
        return $income_heads;
    }

    public function getGroupHead()
    {
        $query = HeadGroup::select('name', 'id')->where(['status' => 'active']);
        $headGroup = $query->orderBy('id', 'DESC')->get();
        return $headGroup;
    }

    public static function getCurrentAccountingYear()
    {
        $accountYear = AccountingYear::select(
            'accounting_year_id',
            'accounting_year_financial',
            'accounting_start_year',
            'accounting_end_year'
        )
            ->where('status', 'active')
            ->orderBy('accounting_year_id', 'DESC')
            ->first();

        if (!$accountYear) {
            return null;
        }

        $year = [
            'accounting_year_id'        => $accountYear->accounting_year_id,
            'accounting_year_financial' => $accountYear->accounting_year_financial,
            'accounting_start_year'     => $accountYear->accounting_start_year,
            'accounting_end_year'       => $accountYear->accounting_end_year,
            'account_year_prefix'       => $accountYear->accounting_start_year . '_' . $accountYear->accounting_end_year,
        ];

        return (object) $year;
    }

    public static function getYearPrefixedTables()
    {
        $currentYear = self::getCurrentAccountingYear();
        if (!$currentYear) {
            return [];
        }

        $prefix = $currentYear->account_year_prefix;

        return [
            'asset_transaction_record'      => $prefix . '_asset_mapping_transaction',
            'donor_transaction_record'      => $prefix . '_donor_mapping_transaction',
            'journal_transactions'          => $prefix . '_journal_transactions',
            'liability_transaction_record'  => $prefix . '_liability_mapping_transaction',
            'vendor_transaction_record'     => $prefix . '_vendor_mapping_transaction',
            'assets_opening_balance'        => $prefix . '_assets_opening_balance',
            'libilities_opening_balance'    => $prefix . '_libilities_opening_balance',
            'contra_journal_trans_mapping'  => $prefix . '_contra_journal_trans_mapping',
        ];
    }

    public function cashAndBankSumation()
    {
        $getTbles    = self::getYearPrefixedTables();
        $currentYear = self::getCurrentAccountingYear();
        $user = Auth::user();

        $journal_transactions =  $getTbles['journal_transactions'];
        $accounting_year_id   =  $currentYear->accounting_year_id;


        // 1️⃣ Get Brought Forward Values
        $accountBf = DB::table('account_brought_forwards')
            ->where('financial_year_id', $accounting_year_id)
            ->when($user && $user->id !== 1, function ($q) use ($user) {
                $q->where('organisation_id', $user->organisation_id);
            })
            ->select('id', 'cash_in_hand', 'cash_at_bank')
            ->first();


        // If no record found, default to 0 to prevent errors
        $cashInHandStart = $accountBf->cash_in_hand ?? 0;
        $cashAtBankStart = $accountBf->cash_at_bank ?? 0;

        // 2️⃣ Get total income and expense transactions



        $incomeCash = DB::table($journal_transactions)
            ->where('payment_type_id', 1)
            ->where('transcation_type', 'income')
            ->when($user && $user->id !== 1, function ($q) use ($user) {
                $q->where('organisation_id', $user->organisation_id);
            })
            ->sum('transaction_amount');

        $incomeBank = DB::table($journal_transactions)
            ->where('payment_type_id', '>', 1)
            ->where('transcation_type', 'income')
            ->when($user && $user->id !== 1, function ($q) use ($user) {
                $q->where('organisation_id', $user->organisation_id);
            })
            ->sum('transaction_amount');

        $expenseCash = DB::table($journal_transactions)
            ->where('payment_type_id', 1)
            ->where('transcation_type', 'expense')
            ->when($user && $user->id !== 1, function ($q) use ($user) {
                $q->where('organisation_id', $user->organisation_id);
            })
            ->sum('transaction_amount');

        $expenseBank = DB::table($journal_transactions)
            ->where('payment_type_id', '>', 1)
            ->where('transcation_type', 'expense')
            ->when($user && $user->id !== 1, function ($q) use ($user) {
                $q->where('organisation_id', $user->organisation_id);
            })
            ->sum('transaction_amount');


        // 3️⃣ Calculate total balances
        $totalCashInHand = ($cashInHandStart + $incomeCash) - $expenseCash;
        $totalCashAtBank = ($cashAtBankStart + $incomeBank) - $expenseBank;

        // 4️⃣ Return the results
        return [
            'account_bf_id' => $accountBf->id ?? null,
            'account_bf_cash_in_hand_opening' => $cashInHandStart,
            'account_bf_cash_at_bank_opening' => $cashAtBankStart,
            'total_cash_in_hand' => $totalCashInHand,
            'total_cash_at_bank' => $totalCashAtBank,
        ];
    }

    public function getPayment_TypeName_Idwise($type_id)
    {
        if (!$type_id) {
            return null; // No ID provided
        }

        // Fetch a single active payment type record
        $payment_type = PaymentType::select('id', 'name')
            ->where('status', 'active')
            ->where('id', $type_id)
            ->first();

        // Return object or null if not found
        return $payment_type ?: null;
    }

    public function getActiveBanks()
    {
        $banks = Bank::select('id', 'name', 'short_code', 'ifsc_code', 'branch', 'account_number')
            ->where('is_active', true)
            ->orderBy('name', 'ASC')
            ->get();

        return $banks;
    }
}
